/**
 * PDFLink - URL to PDF Converter Demo JavaScript
 * Basic form handling and demo functionality
 */

document.addEventListener('DOMContentLoaded', function() {
    
    // URL validation regex
    const urlRegex = /^(https?:\/\/)?([\da-z\.-]+)\.([a-z\.]{2,6})([\/\w \.-]*)*\/?$/;
    
    // Get form elements (works for both basic and advanced forms)
    const convertForm = document.querySelector('#converter form') || document.getElementById('advanced-converter-form');
    const urlInput = document.getElementById('website-url');
    const convertButton = convertForm ? convertForm.querySelector('button[type="submit"]') : null;
    
    if (convertForm && urlInput && convertButton) {
        
        // Add real-time URL validation
        urlInput.addEventListener('input', function() {
            const url = this.value.trim();
            
            if (url === '') {
                this.style.borderColor = '#e9ecef';
                convertButton.disabled = false;
                return;
            }
            
            if (urlRegex.test(url)) {
                this.style.borderColor = '#28a745';
                convertButton.disabled = false;
            } else {
                this.style.borderColor = '#dc3545';
                convertButton.disabled = true;
            }
        });
        
        // Handle form submission
        convertForm.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const url = urlInput.value.trim();
            
            if (!url) {
                alert('Please enter a valid URL');
                return;
            }
            
            if (!urlRegex.test(url)) {
                alert('Please enter a valid URL format (e.g., https://example.com)');
                return;
            }
            
            // Add http:// if no protocol specified
            let finalUrl = url;
            if (!url.startsWith('http://') && !url.startsWith('https://')) {
                finalUrl = 'https://' + url;
            }
            
            // Demo functionality - show conversion process
            showConversionDemo(finalUrl);
        });
    }
    
    function showConversionDemo(url) {
        // Disable the button and show processing
        if (convertButton) {
            const originalText = convertButton.innerHTML;
            convertButton.innerHTML = '<i class="fa fa-spinner fa-spin"></i> Converting...';
            convertButton.disabled = true;
            
            // Simulate conversion process
            setTimeout(function() {
                // Show success message
                showSuccessMessage(url);
                
                // Reset button
                convertButton.innerHTML = originalText;
                convertButton.disabled = false;
                
                // Clear the form
                urlInput.value = '';
                urlInput.style.borderColor = '#e9ecef';
            }, 3000);
        }
    }
    
    function showSuccessMessage(url) {
        // Create success modal or alert
        const message = `
            ✅ Conversion Complete!
            
            URL: ${url}
            PDF Status: Ready for download
            Links Found: ${Math.floor(Math.random() * 50) + 10}
            File Size: ${(Math.random() * 5 + 1).toFixed(1)} MB
            
            Note: This is a demo. In the full version, your PDF would be ready for download!
        `;
        
        alert(message);
    }
    
    // Add smooth scrolling for anchor links
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
    
    // Add form animation on focus
    if (urlInput) {
        urlInput.addEventListener('focus', function() {
            this.parentElement.style.transform = 'scale(1.02)';
            this.parentElement.style.transition = 'transform 0.3s ease';
        });
        
        urlInput.addEventListener('blur', function() {
            this.parentElement.style.transform = 'scale(1)';
        });
    }
    
});

// Utility function to format URLs
function formatUrl(url) {
    if (!url.startsWith('http://') && !url.startsWith('https://')) {
        return 'https://' + url;
    }
    return url;
}

// Demo data for statistics counter animation
function animateCounters() {
    const counters = document.querySelectorAll('.stat_count');
    
    counters.forEach(counter => {
        const target = parseInt(counter.textContent.replace(/[^\d]/g, ''));
        const increment = target / 100;
        let current = 0;
        
        const timer = setInterval(() => {
            current += increment;
            if (current >= target) {
                current = target;
                clearInterval(timer);
            }
            
            if (counter.textContent.includes('K')) {
                counter.textContent = Math.floor(current / 1000) + 'K';
            } else if (counter.textContent.includes('%')) {
                counter.textContent = Math.floor(current) + '%';
            } else {
                counter.textContent = Math.floor(current);
            }
        }, 20);
    });
}

// Initialize counter animation when page loads
document.addEventListener('DOMContentLoaded', function() {
    setTimeout(animateCounters, 1000);
});
